/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "test_common.h"
#if !MX_OS_WINNT
#include <unistd.h>
#include <sys/time.h>
#include <errno.h>
#else
#include "mx_uni.h"
#endif

#if MX_DEBUG
#define verbose_printf printf
#else
#define verbose_printf if (0) printf
#endif

void
simple(
  mx_endpoint_t ep,
  mx_endpoint_addr_t addr,
  int len)
{
  mx_return_t ret;
  mx_request_t recv_req = NULL;
  mx_request_t send_req = NULL;
  mx_segment_t recv_seg;
  mx_segment_t send_seg;
  mx_request_t returned_req;
  mx_status_t status;
  char *recv_buff;
  char *send_buff;
  uint32_t result;

  verbose_printf("simple started for len =%d\n",len);
  send_buff = (char *) malloc(len);
  recv_buff = (char *) malloc(len);
  insist((send_buff != NULL && recv_buff != NULL));

  recv_seg.segment_ptr = recv_buff;
  recv_seg.segment_length = len;
  ret = mx_irecv(ep, &recv_seg, 1, 0, 0, NULL, &recv_req);
  insist(ret == MX_SUCCESS);

  ret = mx_ipeek(ep, &returned_req, &result);
  insist((ret == MX_SUCCESS) && (result == 0));
  if (returned_req != NULL) {
    printf ("ipeek returns req = %p (send_req = %p, recv_req = %p)\n", 
	    returned_req, send_req, recv_req);
    fflush(stdout);
  }
  insist(returned_req == NULL);

  send_seg.segment_ptr = send_buff;
  send_seg.segment_length = len;
  ret = mx_isend(ep, &send_seg, 1, addr, 0, NULL, &send_req);
  insist(ret == MX_SUCCESS);

  do {
    ret = mx_ipeek(ep, &returned_req, &result);
    insist ((ret == MX_SUCCESS));

    if (result) {
      if (returned_req == send_req) {
	ret = mx_test(ep, &send_req, &status, &result);
	insist((ret == MX_SUCCESS) && (result == 1));
	send_req = NULL;

      } else if (returned_req == recv_req) {
	ret = mx_test(ep, &recv_req, &status, &result);
	insist((ret == MX_SUCCESS) && (result == 1));
	recv_req = NULL;

      } else {
        insist(((returned_req == send_req) || (returned_req == recv_req)));
      }
    }
  } while (send_req != NULL || recv_req != NULL);
  free(recv_buff);
  free(send_buff);
  verbose_printf("simple completed\n");
}


void
harder(
  mx_endpoint_t ep,
  mx_endpoint_addr_t addr,
  int len,
  int nreq)
{
  mx_return_t ret;
  mx_request_t *recv_req;
  mx_request_t *send_req;
  mx_segment_t recv_seg;
  mx_segment_t send_seg;
  mx_request_t returned_req;
  mx_status_t status;
  char *recv_buff;
  char *send_buff;
  uint32_t result;
  int nleft;
  int i;

  verbose_printf("harder started for len =%d\n",len);
  send_buff = (char *) malloc(len);
  recv_buff = (char *) malloc(len);
  insist((send_buff != NULL && recv_buff != NULL));

  recv_req = (mx_request_t *) calloc (nreq, sizeof(mx_request_t));
  send_req = (mx_request_t *) calloc (nreq, sizeof(mx_request_t));
  insist((recv_req != NULL && send_req != NULL));

  send_seg.segment_ptr = send_buff;
  send_seg.segment_length = len;

  /* blast off the sends */
  for (i=0; i<nreq; ++i) {
    ret = mx_isend(ep, &send_seg, 1, addr, 0,
		   (void *)(uintptr_t)i, &send_req[i]);
    insist(ret == MX_SUCCESS);
  }
 
  recv_seg.segment_ptr = recv_buff;
  recv_seg.segment_length = len;

  /* now the receives */
  for (i=0; i<nreq; ++i) {
    ret = mx_irecv(ep, &recv_seg, 1, 0, 0,
                   (void *)(uintptr_t)(nreq+i), &recv_req[i]);
    insist(ret == MX_SUCCESS);
  }

  nleft = nreq*2;
  while (nleft > 0) {

    ret = mx_ipeek(ep, &returned_req, &result);
    insist ((ret == MX_SUCCESS));

    if (result) {
      mx_request_t req_copy = returned_req;
      ret = mx_test(ep, &req_copy, &status, &result);
      insist((ret == MX_SUCCESS) && (result == 1));


      i = (uintptr_t) status.context;
      insist ((i >= 0 && i < 2*nreq));

      /* send ? */
      if (i < nreq) {
	insist((returned_req == send_req[i]));
	send_req[i] = NULL;

      } else {
        i -= nreq;
	insist((returned_req == recv_req[i]));
	recv_req[i] = NULL;
      }

      --nleft;
    }
  }

  free(recv_buff);
  free(send_buff);
  free(recv_req);
  free(send_req);
  verbose_printf("harder completed\n");
}

void
hardest(
  mx_endpoint_t ep,
  mx_endpoint_addr_t addr,
  int len,
  int nreq)
{
  mx_return_t ret;
  mx_request_t *recv_req;
  mx_request_t *send_req;
  mx_segment_t recv_seg;
  mx_segment_t send_seg;
  mx_request_t returned_req;
  mx_status_t status;
  char *recv_buff;
  char *send_buff;
  uint32_t result;
  int nleft;
  int i;

  verbose_printf("hardest started for len =%d\n",len);
  send_buff = (char *) malloc(len);
  recv_buff = (char *) malloc(len);
  insist((send_buff != NULL && recv_buff != NULL));

  recv_req = (mx_request_t *) calloc (nreq, sizeof(mx_request_t));
  send_req = (mx_request_t *) calloc (nreq, sizeof(mx_request_t));
  insist((recv_req != NULL && send_req != NULL));

  send_seg.segment_ptr = send_buff;
  send_seg.segment_length = len;

  /* blast off the sends */
  for (i=0; i<nreq; ++i) {
    ret = mx_isend(ep, &send_seg, 1, addr, 0,
		   (void *)(uintptr_t)i, &send_req[i]);
    insist(ret == MX_SUCCESS);
  }

  /* wait for them all */
  nleft = nreq;
  while (nleft > 0) {

    ret = mx_ipeek(ep, &returned_req, &result);
    insist ((ret == MX_SUCCESS));

    if (result) {
      mx_request_t req_copy = returned_req;
      ret = mx_test(ep, &req_copy, &status, &result);
      insist((ret == MX_SUCCESS) && (result == 1));


      i = (uintptr_t) status.context;
      insist ((i >= 0 && i < nreq));

      insist((returned_req == send_req[i]));
      send_req[i] = NULL;

      --nleft;
    }
  }
 
  recv_seg.segment_ptr = recv_buff;
  recv_seg.segment_length = len;

  /* now the receives */
  for (i=0; i<nreq; ++i) {
    ret = mx_irecv(ep, &recv_seg, 1, 0, 0,
                   (void *)(uintptr_t)i, &recv_req[i]);
    insist(ret == MX_SUCCESS);
  }

  /* wait for them all */
  nleft = nreq;
  while (nleft > 0) {

    ret = mx_ipeek(ep, &returned_req, &result);
    insist ((ret == MX_SUCCESS));

    if (result) {
      mx_request_t req_copy = returned_req;
      ret = mx_test(ep, &req_copy, &status, &result);
      insist((ret == MX_SUCCESS) && (result == 1));


      i = (uintptr_t) status.context;
      insist ((i >= 0 && i < nreq));

      insist((returned_req == recv_req[i]));
      recv_req[i] = NULL;

      --nleft;
    }
  }

  free(recv_buff);
  free(send_buff);
  free(recv_req);
  free(send_req);
  verbose_printf("hardest completed\n");
}

static void
usage(void)
{
  fprintf(stderr, "mx_ipeek_test [ -b <board_num> ] [ -e <eid> ]\n");
  exit(1);
}

int
main(int argc, char *argv[])
{
  int c;
  mx_return_t ret;
  mx_endpoint_t ep;
  mx_endpoint_addr_t addr;

  uint32_t board_id = MX_ANY_NIC;
  uint32_t my_eid = MX_ANY_ENDPOINT;

  while ((c = getopt(argc, argv, "b:e:h")) != -1) switch(c) {
  case 'b':
    board_id = atoi(optarg);
    break;
  case 'e':
    my_eid = atoi(optarg);
    break;
  default:
    usage();
  }
  ret = mx_init();
  insist(ret == MX_SUCCESS);

  ret = mx_open_endpoint(board_id, my_eid, 0, NULL, 0, &ep);
  insist(ret == MX_SUCCESS);

  ret = mx_get_endpoint_addr(ep, &addr);
  insist(ret == MX_SUCCESS);


  simple(ep, addr, 80);
  simple(ep, addr, 10000);
  simple(ep, addr, 40000);
  harder(ep, addr, 80, 30);
  harder(ep, addr, 10000, 30);
  harder(ep, addr, 40000, 30);
  harder(ep, addr, 80, 100);
  harder(ep, addr, 10000, 100);
  hardest(ep, addr, 80, 30);
  hardest(ep, addr, 10000, 30);
  hardest(ep, addr, 80, 100);
  hardest(ep, addr, 10000, 100);
  /*  hardest(ep, addr, 40000, 30); */

  ret = mx_close_endpoint(ep);
  insist(ret == MX_SUCCESS);

  ret = mx_finalize();
  insist(ret == MX_SUCCESS);

  return 0;
}
